# scripts/download_cfhtlens.py
import sys, time
import pyvo
from astropy.table import Table

TAP_URL = "https://www.cadc-ccda.hia-iha.nrc-cnrc.gc.ca/tap"

def main():
    svc = pyvo.dal.TAPService(TAP_URL)

    # 0) Check that the table exists and list useful columns
    cols = svc.run_sync("""
        SELECT table_schema, table_name, column_name, datatype
        FROM TAP_SCHEMA.columns
        WHERE table_schema='cfht' AND table_name='clens'
        ORDER BY column_name
    """).to_table()
    print(f"[info] Found {len(cols)} columns in cfht.clens")
    print(cols[0:15])  # show a few to verify names

    # 1) Grab a small sample to see real column names/values
    sample = svc.run_sync("""
        SELECT TOP 100000
          id, field,
          ALPHA_J2000 AS ra_deg,
          DELTA_J2000 AS dec_deg,
          e1, e2, weight, m, c2,
          Z_B, ODDS,
          A_WORLD, B_WORLD, FLUX_RADIUS,
          LP_log10_SM_MED AS Mstar_log10,
          star_flag
        FROM cfht.clens
        WHERE star_flag = 0
    """).to_table()
    Table(sample).write("data/CFHTLenS_sample.csv", format="csv", overwrite=True)
    print(f"[info] Wrote data/CFHTLenS_sample.csv ({len(sample)} rows)")

    # 2) Submit a full async job (remove TOP; keep star cut)
    adql_full = """
        SELECT
          id, field,
          ALPHA_J2000 AS ra_deg,
          DELTA_J2000 AS dec_deg,
          e1, e2, weight, m, c2,
          Z_B, ODDS,
          A_WORLD, B_WORLD, FLUX_RADIUS,
          LP_log10_SM_MED AS Mstar_log10,
          star_flag
        FROM cfht.clens
        WHERE star_flag = 0
    """
    job = svc.submit_job(adql_full, maxrec=-1)
    job.run()
    while job.phase not in ("COMPLETED","ERROR","ABORTED"):
        time.sleep(3)
    if job.phase != "COMPLETED":
        print(f"[error] TAP job failed: {job.phase}")
        sys.exit(2)

    full = job.fetch_result().to_table()
    Table(full).write("data/CFHTLenS_full.csv", format="csv", overwrite=True)
    print(f"[info] Wrote data/CFHTLenS_full.csv ({len(full)} rows)")

if __name__ == "__main__":
    main()

